/*:ja
 * @target MZ
 * @plugindesc オリジナル戦闘「フローシステム」と、連携する立ち絵表示プラグイン
 * @author 柊 七紀, Gemini
 * @help
 * フローシステムに加え、戦闘中のHPや状況に応じてアクターの立ち絵・表情を
 * 自動で切り替える機能を提供します。
 *
 * --- 立ち絵・表情差分機能の使い方 ---
 *
 * ■ ファイル準備
 * 画像ファイルはすべて img/pictures/戦闘 フォルダに配置してください。
 * ファイル名は以下のように設定します。
 *
 * ・本体: (例) citrus_body.png
 * ・衣装: (例) c_1_1.png
 * ・衣装(中破): (例) c_1_1_middle.png
 * ・表情(通常): (例) face_normal.png
 * ・表情(ダメージ): (例) face_damage.png
 * ・表情(羞恥): (例) face_shame.png
 *
 * ■ プラグインパラメータの設定
 * このプラグインのパラメータで、各画像ファイルの基本名や、
 * 状態が切り替わるHPの割合などを設定してください。
 *
 * ・Target Actor Id: 立ち絵を表示する対象のアクターID (例: 1)
 * ・Costume Variable Id: 衣装名を格納する変数ID (例: 5)
 *
 * 変数5番に "c_1_1" という文字列を入れておくと、
 * 衣装は "c_1_1.png", 中破時は "c_1_1_middle.png" が読み込まれます。
 *
 * --- フローシステムの使い方 ---
 *
 * アクターのメモ欄に <no_flow> と記述すると、そのアクターは
 * フローシステムを使用しなくなります。
 *　↑ <no_flow>は使用しない方向で。
 * 
 * @param ---立ち絵設定---
 * @default
 *
 * @param targetActorId
 * @text 対象アクターID
 * @desc 立ち絵を表示するアクターのID。
 * @type actor
 * @default 1
 *
 * @param costumeVariableId
 * @text 衣装変数ID
 * @desc 衣装のファイル名を管理する変数のID。
 * @type variable
 * @default 5
 *
 * @param bodyImage
 * @text 本体画像ファイル名
 * @desc 立ち絵の本体部分の画像ファイル名。(例: citrus_body)
 * @type string
 * @default citrus_body
 *
 * @param faceNormalImage
 * @text 通常時の表情
 * @desc 通常時の表情ファイル名。(例: face_normal)
 * @type string
 * @default face_normal
 *
 * @param faceDamageImage
 * @text ダメージ時の表情
 * @desc ダメージを受けた時の表情ファイル名。(例: face_damage)
 * @type string
 * @default face_damage
 *
 * @param faceShameImage
 * @text 羞恥時の表情
 * @desc HPが中破ラインを下回った時の表情ファイル名。(例: face_shame)
 * @type string
 * @default face_shame
 *
 * @param middleHpRate
 * @text 中破HP割合
 * @desc このHP割合(%)以下になると衣装が中破状態になります。
 * @type number
 * @default 50
 *
 * @param lowHpRate
 * @text 大破HP割合
 * @desc このHP割合(%)以下になると衣装が非表示(大破)になります。
 * @type number
 * @default 20
 *
 * @param damageFaceDuration
 * @text ダメージ表情の表示時間
 * @desc ダメージを受けた表情を表示しておくフレーム数。(60F = 1秒)
 * @type number
 * @default 60
 *
 * @command SetStandSpriteVisibility
 * @text 立ち絵・アイコンの可視状態変更
 * @desc バトル中の立ち絵、アイコン背景、フローアイコンの表示/非表示を切り替えます。
 * @arg standSpriteVisibility
 * @text 立ち絵
 * @desc 'show'で表示、'hide'で非表示にします。
 * @type select
 * @option 表示
 * @value show
 * @option 非表示
 * @value hide
 * @default show
 * @arg iconBackgroundVisibility
 * @text アイコン背景
 * @desc 'show'で表示、'hide'で非表示にします。
 * @type select
 * @option 表示
 * @value show
 * @option 非表示
 * @value hide
 * @default show
 * @arg flowIconsVisibility
 * @text フローアイコン
 * @desc 'show'で表示、'hide'で非表示にします。
 * @type select
 * @option 表示
 * @value show
 * @option 非表示
 * @value hide
 * @default show
 */

(() => {
    'use strict';
    const pluginName = 'FlowSystem';
    const params = PluginManager.parameters(pluginName);

    const TargetActorId = parseInt(params.targetActorId, 10);
    const CostumeVariableId = parseInt(params.costumeVariableId, 10);
    const BodyImage = params.bodyImage;
    const FaceNormalImage = params.faceNormalImage;
    const FaceDamageImage = params.faceDamageImage;
    const FaceShameImage = params.faceShameImage;
    const MiddleHpRate = parseInt(params.middleHpRate, 10);
    const LowHpRate = parseInt(params.lowHpRate, 10);
    const DamageFaceDuration = parseInt(params.damageFaceDuration, 10);
    const FINISH_SKILL_HISTORY_VARIABLE_ID = 50;

//-----------------------------------------------------------------------------
    // プラグインコマンド
    //-----------------------------------------------------------------------------
    PluginManager.registerCommand(pluginName, "SetStandSpriteVisibility", args => {
        const standSpriteVisibility = args.standSpriteVisibility;
        const iconBackgroundVisibility = args.iconBackgroundVisibility;
        const flowIconsVisibility = args.flowIconsVisibility;
        const scene = SceneManager._scene;

        if (scene instanceof Scene_Battle) {
            // 立ち絵の可視状態変更
            if (scene._standSpriteBody && scene._standSpriteCostume && scene._standSpriteFace) {
                const isShow = standSpriteVisibility === "show";
                scene._standSpriteBody.visible = isShow;
                scene._standSpriteCostume.visible = isShow;
                scene._standSpriteFace.visible = isShow;
                // [新機能] 歩行グラフィックも表示/非表示を切り替える
                if (scene._flowActorSprite) {
                    scene._flowActorSprite.visible = isShow;
                }
                if (isShow) {
                    scene.updateStandSprite();
                    scene.updateFlowActorSprite(); // 表示する場合は更新処理も呼び出す
                }
            }
            // アイコン背景の可視状態変更
            if (scene._flowIconBackgroundSprite) {
                scene._flowIconBackgroundSprite.visible = (iconBackgroundVisibility === "show");
            }
            // フローアイコンの可視状態変更
            if (scene.iconSprites) {
                scene.iconSprites.forEach(sprite => {
                    if (sprite) sprite.visible = (flowIconsVisibility === "show");
                });
            }
        }
    });
    
    //-----------------------------------------------------------------------------
    // Game_Battler
    // フローシステムのコアロジック
    //-----------------------------------------------------------------------------

    Game_Battler.prototype.isFlowAllowed = function() {
        if (!this.isActor()) return false;
        return !this.actor().meta.no_flow;
    };

    const _Game_Battler_initMembers = Game_Battler.prototype.initMembers;
    Game_Battler.prototype.initMembers = function() {
        _Game_Battler_initMembers.call(this);
        this.clearFlow();
    };
    
    const _Game_Battler_onBattleEnd = Game_Battler.prototype.onBattleEnd;
    Game_Battler.prototype.onBattleEnd = function() {
        _Game_Battler_onBattleEnd.call(this);
        this.clearFlow();
    };

    const _Game_Battler_useItem = Game_Battler.prototype.useItem;
    Game_Battler.prototype.useItem = function(item) {
        _Game_Battler_useItem.call(this, item);

        if ($gameParty.inBattle() && this.isFlowAllowed()) {
            const scene = SceneManager._scene;
            scene.setCurrentFlowActor(this); // [新機能] フロー所有者を設定

            const memo = item.note;
            const flowMatches = memo.match(/<flow:(.+?):(\d+)>/g);

            if (flowMatches) {
                for (let i = 0; i < flowMatches.length; i++) {
                    const flowMatch = flowMatches[i].match(/<flow:(.+?):(\d+)>/);
                    this.flowWords.push(flowMatch[1]);
                    this.flowIcons.push(Number(flowMatch[2]));
                }
                this.showIconEffects();
            } else {
                this.clearFlow();
            }
            
            if (DataManager.isSkill(item) && [1, 3, 4, 5, 6].includes(item.scope) && BattleManager._targets && BattleManager._targets[0] && !this._isFlowFinishing) {
                this._lastFlowTargetIndex = BattleManager._targets[0].index();
            } else if (!this._isFlowFinishing) {
                this._lastFlowTargetIndex = -1;
            }
            this.finishCheck();
        }
    };

    Game_Battler.prototype.clearFlow = function() {
        this.flowWords = [];
        this.flowIcons = [];
        this._lastFlowTargetIndex = -1;
        if (SceneManager._scene instanceof Scene_Battle) {
            SceneManager._scene.clearIconEffects();
            SceneManager._scene.setCurrentFlowActor(null); // [新機能] フロー所有者をクリア
        }
    };

    Game_Battler.prototype.showIconEffects = function() {
        const battleScene = SceneManager._scene;
        if (battleScene instanceof Scene_Battle) {
            battleScene.clearIconEffects();
            for (let i = 0; i < this.flowIcons.length; i++) {
                const iconIndex = this.flowIcons[i];
                const iconX = 626 + i * 50;
                const iconY = 383;
                battleScene.createIconSprite(iconIndex, iconX, iconY);
            }
        }
    };

    Game_Battler.prototype.finishCheck = function() {
        const finishPatterns = {
            // ―――――【基本】―――――
            // 疾風刃
            "slash,slash,slash": 31,
            // ハートブレイカー
            "slash,slash,thrust": 25,
            // 打ち崩し
            "slash,slash,smash": 33,
            // サベージスラスト
            "slash,thrust,slash": 34,
            // スパークラプチャー
            "slash,smash,slash": 80,
            // ピアース
            "slash,thrust,thrust": 26,
            // スタンピードデブリ
            "slash,smash,smash": 81,
            // 昇光の奔流
            "slash,thrust,smash": 82,
            // 堕天の暗雲
            "slash,smash,thrust": 83,
            // 連続突き
            "thrust,thrust,thrust": 53,
            // フレイムレイジ
            "thrust,thrust,slash": 84,
            // ピアッシング・ロア
            "thrust,thrust,smash": 85,
            // ランページ
            "thrust,slash,thrust": 86,
            // デザート・アサルト
            "thrust,smash,thrust": 87,
            // クロス・スラッシュ
            "thrust,slash,slash": 88,
            // スプラッシュ
            "thrust,smash,smash": 89,
            // 破滅の天秤
            "thrust,slash,smash": 90,
            // ミラージュエッジ
            "thrust,smash,slash": 91,
            // 崩落の一撃
            "smash,smash,smash": 92,
            // タイダルスラッシャー
            "smash,smash,slash": 93,
            // 真空撃
            "smash,smash,thrust": 94,
            // パニッシュメント
            "smash,slash,smash": 95,
            // ダークコネクト
            "smash,thrust,smash": 96,
            // タイタンプレッシャー
            "smash,slash,slash": 97,
            // グレイブテイカー
            "smash,thrust,thrust": 98,
            // ブラッドライン
            "smash,slash,thrust": 99,
            // ステップシャドウ
            "smash,thrust,slash": 220,

            // ―――――【マジック】―――――
            // ブレイジング・トレース
            "magic,slash": 29,
            // メテオストーム
            "magic,thrust,thrust": 221,
            // フレイムレイル
            "magic,thrust,smash": 222,
            // 紅蓮龍
            "magic,thrust,slash": 223,
            // バーンアウト
            "magic,smash": 224,
            // 残滓回収
            "magic,buff": 225,
            // エレメントバースト
            "magic,magic,magic": 226,
            // 魔力の斬撃
            "magic,magic,slash": 227,
            // 魔力の刺突
            "magic,magic,thrust": 228,
            // 魔力の打撃
            "magic,magic,smash": 229,
            // 魔刃瞬連斬
            "magic,rush": 230,
            // 魔導深淵
            "magic,magic,buff": 231,
            // スパークボール
            "slash,magic": 232,
            // ブラックアウト
            "thrust,magic": 233,
            // セイレーン
            "smash,magic": 234,

            // ―――――【ブースト】―――――
            // ハートフル・スリー
            "buff,buff": 235,
            // ウォークライ
            "buff,magic": 236,
            // 威圧感
            "buff,smash": 237,
            // 獅子奮迅
            "buff,slash": 238,
            // ムーブゲイザー
            "buff,thrust": 239,
            // 超越
            "buff,rush": 240,
            // ビートアップ
            "buff,drain": 241,
            // 完全超越
            "rush,buff,drain": 242,


            // ―――――【ラッシュ】―――――
            // 神速乱舞
            "rush,rush,rush": 243,
            // スピンスラッシュ
            "rush,slash": 244,
            // ジョストスラスト
            "rush,thrust": 245,
            // ギガントスマッシュ
            "rush,smash": 246,
            // ラスターカノン
            "rush,magic": 247,
            // キルマーク
            "rush,drain": 248,

            // ―――――【ドレイン】―――――
            // ハンティングエリア
            "drain,drain,drain": 249,
            // ヴォイドスラッシュ
            "drain,slash": 250,
            // ファングスラスト
            "drain,thrust": 251,
            // デモンズスマッシュ
            "drain,drain,smash": 252,
            // ブラックストリーム
            "drain,magic,drain": 253,
            // 百鬼夜行
            "drain,drain,rush": 254,


            // ―――――【セクシー】―――――
            // ラスト・エクスプロージョン
            "sexy,sexy,sexy": 255,
            // スレイブチェイン
            "sexy,sexy,magic": 256,
            // 性の解放
            "sexy,sexy,buff": 257,


            // ―――――【ハンド・レッグ】―――――
            // 荒嵐拳
            "hand,hand,hand": 67,
            // 跳び回し蹴り
            "leg,leg,leg": 66,
            // 舞踊連脚
            "buff,leg": 69,
            // 流気操
            "hand,magic": 306,
            // 破城蹴り
            "hand,hand,leg": 307,
            // 破天突き
            "leg,hand,leg": 308,
            // 龍歩
            "leg,leg,hand": 309,
            // 双牙
            "hand,leg,hand": 310,


            // カンナ
            "kanna_sl,kanna_th,kanna_sm": 168,
            "kanna_sl,kanna_sl,kanna_th": 169,
            "kanna_sl,kanna_sm,kanna_th": 168,
            "kanna_th,kanna_sl": 168,
            "kanna_th,kanna_th": 169,
            "kanna_th,kanna_sm": 168,
            "kanna_sm,kanna_sl,kanna_sl": 168,
            "kanna_sm,kanna_sl,kanna_th": 169,
            "kanna_sm,kanna_th,kanna_sl": 169,
            "kanna_sm,kanna_sm": 168,
            "kanna_sl,kanna_sl": 169,
            "kanna_ru": 170,
            "kanna_sl,kanna_ru": 170,
            "kanna_sl,kanna_sl,kanna_ru": 170,
            "kanna_th,kanna_ru": 170,
            "kanna_th,kanna_th,kanna_ru": 170,
            "kanna_sm,kanna_ru": 170,
            "kanna_sm,kanna_sm,kanna_ru": 170,

            // エナ
            "e_mag,e_mag": 105,
            "e_mag2,e_mag2": 107,
            "e_mag,e_mag2": 107,
            "e_mag2,e_mag": 105,

            // マリーズ
            "ma_t,ma_t,ma_t": 141,
            "ma_t,ma_b": 142,
            "ma_b,ma_t": 142,
            "ma_t,ma_t,ma_b": 142,

            // ノエル
            "no_1,no_2,no_2": 151,
            "no_2,no_1,no_2": 150,
            "no_2,no_2,no_1": 151,
            "no_2,no_2,no_2": 150,

            // ナツキ
            "natu_stance,natu_stance,natu_t": 159,
            "natu_stance,natu_stance,natu_s": 159,
            "natu_t,natu_t,natu_t": 160,
            "natu_s,natu_s,natu_s": 160,

            // シャオリン
            "sr_p,sr_p,sr_p": 303,
            "sr_p,sr_p,sr_k": 303,
            "sr_k,sr_p,sr_p": 303,
            "sr_k,sr_k,sr_k": 304,
            "sr_k,sr_k,sr_p": 304,
            "sr_p,sr_k,sr_k": 304,
            "sr_b,sr_p": 305,
            "sr_b,sr_k": 305,
            "sr_p,sr_b": 305,
            "sr_k,sr_b": 305,
            "sr_p,sr_p,sr_b": 305,
            "sr_k,sr_k,sr_b": 305,
            "sr_p,sr_k,sr_p": 305,
            "sr_k,sr_p,sr_k": 305,
            
            // カレン
            "kr_sl,kr_sl": 319,
            "kr_sm,kr_sl,kr_sl": 320,
            "kr_sl,kr_sm,kr_sm": 320,
            "kr_sl,kr_sm,kr_sl": 320,
            "kr_sm,kr_sm,kr_sl": 320,
            "kr_sm,kr_sm,kr_sm": 320,
            "kr_sm,kr_sl,kr_sm": 320,

            // ローゼス
            "ro_t,ro_s": 125,
            "ro_t,ro_d": 126
            
        };

        const flowString = this.flowWords.join(",");
        if (finishPatterns[flowString] && finishPatterns[flowString] !== 1) {
            const finishSkillId = finishPatterns[flowString];
            if (SceneManager._scene._logWindow) {
                SceneManager._scene._logWindow.addText("――Flow Finish――");
            }
            AudioManager.playSe({ name: 'Flash1', volume: ConfigManager.seVolume, pitch: 100, pan: 0 });
            this.flowFinish(finishSkillId);
            setTimeout(() => this.clearFlow(), 2000);
        } else if (this.flowWords.length >= 3) {
            setTimeout(() => this.clearFlow(), 2000);
        }
    };
    
    Game_Battler.prototype._isFlowFinishing = false;
    Game_Battler.prototype.flowFinish = function(finishID) {
        this._isFlowFinishing = true;

        const actor = this;
        if (!actor) {
            this._isFlowFinishing = false;
            return;
        }

        const action = new Game_Action(actor, true);
        action.setSkill(finishID);

        const skill = $dataSkills[finishID];
        let targetIndex = -1;
        let actionIsValid = false;

        if (skill) {
            if ([1, 3, 4, 5, 6].includes(skill.scope)) {
                if (this._lastFlowTargetIndex >= 0 && $gameTroop.members()[this._lastFlowTargetIndex]?.isAlive()) {
                    targetIndex = this._lastFlowTargetIndex;
                } else {
                    const enemies = $gameTroop.aliveMembers();
                    if (enemies.length > 0) targetIndex = enemies[0].index();
                }
                if (targetIndex !== -1) actionIsValid = true;
            } else if ([2, 8, 10, 13, 14].includes(skill.scope)) {
                actionIsValid = true;
            } else if ([7, 9, 11, 12].includes(skill.scope)) {
                targetIndex = actor.index();
                actionIsValid = true;
            }

            if (!BattleManager.isBattleEnd() && actionIsValid) {
                if (targetIndex !== -1) action.setTarget(targetIndex);
                BattleManager.forceAction(actor);
                actor._actions.push(action);
            }
        }
        
        if (this.isActor() && this.actorId() === TargetActorId) {
            const history = $gameVariables.value(FINISH_SKILL_HISTORY_VARIABLE_ID) || [];
            if (!history.includes(finishID)) {
                history.push(finishID);
                $gameVariables.setValue(FINISH_SKILL_HISTORY_VARIABLE_ID, history);
            }
        }
        
        this._isFlowFinishing = false;
    };

    const _Game_Battler_performDamage = Game_Battler.prototype.performDamage;
    Game_Battler.prototype.performDamage = function() {
        _Game_Battler_performDamage.call(this);
        if (this.isActor() && this.actorId() === TargetActorId) {
            const scene = SceneManager._scene;
            if (scene instanceof Scene_Battle && scene.isStandSpriteReady) {
                scene.setDamageFace();
            }
        }
    };

    //-----------------------------------------------------------------------------
    // BattleManager
    // [新機能] 行動開始時にアクターをシーンに通知する
    //-----------------------------------------------------------------------------
    const _BattleManager_startAction = BattleManager.startAction;
    BattleManager.startAction = function() {
        const subject = this._subject;
        _BattleManager_startAction.call(this);
        if (SceneManager._scene instanceof Scene_Battle && subject.isActor()) {
            SceneManager._scene.setCurrentFlowActor(subject);
        }
    };

    //-----------------------------------------------------------------------------
    // Scene_Battle
    // 立ち絵・表情差分、フローアイコンの管理
    //-----------------------------------------------------------------------------
    const _Scene_Battle_create = Scene_Battle.prototype.create;
    Scene_Battle.prototype.create = function() {
        _Scene_Battle_create.call(this);
        this.iconSprites = [];
        this._damageFaceDuration = 0;
        this.isStandSpriteReady = false;
        this._currentFlowActor = null; // [新機能] フロー所有者を管理
        this._flowActorSpriteCharacterName = ""; // [新機能] 画像キャッシュ用
        this.displayFlowIconBackground();
        this.displayActorStandImage();
        this.createFlowActorSprite(); // [新機能] 歩行グラフィック用のスプライトを作成
    };

    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function() {
        _Scene_Battle_update.call(this);
        if (this.isStandSpriteReady) {
            this.updateStandSprite();
        }
    };

    Scene_Battle.prototype.displayActorStandImage = function() {
        this._standSpriteBody = new Sprite();
        this._standSpriteCostume = new Sprite();
        this._standSpriteFace = new Sprite();
        
        [this._standSpriteBody, this._standSpriteCostume, this._standSpriteFace].forEach(sprite => {
            sprite.anchor.x = 0.5;
            sprite.anchor.y = 1;
            sprite.x = Graphics.width / 4 + 16;
            sprite.y = Graphics.height;
            sprite.scale.set(0.7);
            this._spriteset._pictureContainer.addChild(sprite);
        });

        this._standSpriteBody.zIndex = -1;
        this._standSpriteCostume.zIndex = -2;
        this._standSpriteFace.zIndex = 0;

        this._standSpriteBody.bitmap = ImageManager.loadPicture(`戦闘/${BodyImage}`);
        
        this.isStandSpriteReady = true;
        this.updateStandSprite();
    };

    Scene_Battle.prototype.updateStandSprite = function() {
        const actor = $gameActors.actor(TargetActorId);
        if (!actor || !this._standSpriteBody.visible) return;
        
        const hpRate = actor.hpRate() * 100;
        

        // 戦闘テストなど、変数が初期値の場合の処理
        if ($gameVariables.value(CostumeVariableId) == 0) {
            $gameVariables.setValue(CostumeVariableId, "シトラスの衣装");
        }
        const costumeName = $gameVariables.value(CostumeVariableId);



        
        let costumeFile = `戦闘/${costumeName}`;
        if (hpRate <= LowHpRate) {
            this._standSpriteCostume.visible = false;
        } else if (hpRate <= MiddleHpRate) {
            costumeFile += "_middle";
            this._standSpriteCostume.visible = true;
        } else {
            this._standSpriteCostume.visible = true;
        }

        if (this._standSpriteCostume.visible && this._currentCostumeFile !== costumeFile) {
            this._currentCostumeFile = costumeFile;
            this._standSpriteCostume.bitmap = ImageManager.loadPicture(costumeFile);
        }

        if (this._damageFaceDuration > 0) {
            this._damageFaceDuration--;
        } else {
            let faceFile = `戦闘/`;
            if (hpRate <= MiddleHpRate) {
                faceFile += FaceShameImage;
            } else {
                faceFile += FaceNormalImage;
            }
            if (this._currentFaceFile !== faceFile) {
                this._currentFaceFile = faceFile;
                this._standSpriteFace.bitmap = ImageManager.loadPicture(faceFile);
            }
        }
    };

    Scene_Battle.prototype.setDamageFace = function() {
        const faceFile = `戦闘/${FaceDamageImage}`;
        if (this._currentFaceFile !== faceFile) {
            this._currentFaceFile = faceFile;
            this._standSpriteFace.bitmap = ImageManager.loadPicture(faceFile);
        }
        this._damageFaceDuration = DamageFaceDuration;
    };
    
    Scene_Battle.prototype.displayFlowIconBackground = function() {
        this._flowIconBackgroundSprite = new Sprite(ImageManager.loadPicture('戦闘/flow_icon_background'));
        this._flowIconBackgroundSprite.x = 620;
        this._flowIconBackgroundSprite.y = 360;
        this._flowIconBackgroundSprite.zIndex = 2;
        this.addChild(this._flowIconBackgroundSprite);
    };

    Scene_Battle.prototype.createIconSprite = function(iconIndex, x, y) {
        const sprite = new Sprite(ImageManager.loadSystem('IconSet'));
        const pw = ImageManager.iconWidth;
        const ph = ImageManager.iconHeight;
        const sx = iconIndex % 16 * pw;
        const sy = Math.floor(iconIndex / 16) * ph;
        sprite.setFrame(sx, sy, pw, ph);
        sprite.move(x, y);
        this.addChild(sprite);
        this.iconSprites.push(sprite);
    };

    Scene_Battle.prototype.clearIconEffects = function() {
        if (this.iconSprites) {
            this.iconSprites.forEach(sprite => this.removeChild(sprite));
        }
        this.iconSprites = [];
    };

    const _Scene_Battle_startPartyCommandSelection = Scene_Battle.prototype.startPartyCommandSelection;
    Scene_Battle.prototype.startPartyCommandSelection = function() {
        _Scene_Battle_startPartyCommandSelection.call(this);
        this.refreshFlowIconsForPartyCommand();
        this.setCurrentFlowActor($gameParty.leader()); // [新機能] パーティコマンド時はリーダーを表示
    };

    // [新機能] アクターコマンド選択開始時にフロー所有者を更新
    const _Scene_Battle_startActorCommandSelection = Scene_Battle.prototype.startActorCommandSelection;
    Scene_Battle.prototype.startActorCommandSelection = function() {
        _Scene_Battle_startActorCommandSelection.call(this);
        this.setCurrentFlowActor(BattleManager.actor());
    };

    Scene_Battle.prototype.refreshFlowIconsForPartyCommand = function() {
        this.clearIconEffects();
        const protagonist = $gameActors.actor(TargetActorId);
        if (protagonist && protagonist.flowIcons && protagonist.flowIcons.length > 0) {
            for (let i = 0; i < protagonist.flowIcons.length; i++) {
                const iconIndex = protagonist.flowIcons[i];
                const iconX = 626 + i * 50;
                const iconY = 383;
                this.createIconSprite(iconIndex, iconX, iconY);
            }
        }
    };

    // [新機能] フロー所有者キャラクターのスプライトを作成
    Scene_Battle.prototype.createFlowActorSprite = function() {
        this._flowActorSprite = new Sprite();
        this._flowActorSprite.anchor.x = 0.5;
        this._flowActorSprite.anchor.y = 1;
        this._flowActorSprite.x = 790; // flow_icon_background の右側あたり
        this._flowActorSprite.y = 415;
        this._flowActorSprite.visible = false;
        this.addChild(this._flowActorSprite);
    };
    
    // [新機能] 現在のフロー所有者を設定し、スプライトを更新する
    Scene_Battle.prototype.setCurrentFlowActor = function(actor) {
        if (this._currentFlowActor !== actor) {
            this._currentFlowActor = actor;
            this.updateFlowActorSprite();
        }
    };
    
    // [新機能] フロー所有者のスプライト表示を更新する
    Scene_Battle.prototype.updateFlowActorSprite = function() {
        const actor = this._currentFlowActor;
        if (actor && actor.isActor() && actor.characterName()) {
            const charName = actor.characterName();
            const bitmap = ImageManager.loadCharacter(charName);
            this._flowActorSprite.bitmap = bitmap;

            if (bitmap.isReady()) {
                this.setFlowActorSpriteFrame(actor);
            } else {
                bitmap.addLoadListener(() => {
                    if (this._currentFlowActor === actor) {
                        this.setFlowActorSpriteFrame(actor);
                    }
                });
            }
        } else {
            this._flowActorSprite.visible = false;
        }
    };

    Scene_Battle.prototype.setFlowActorSpriteFrame = function(actor) {
        if (!this._flowActorSprite.bitmap) return;
        const charName = actor.characterName();
        const big = ImageManager.isBigCharacter(charName);
        const pw = this._flowActorSprite.bitmap.width / (big ? 3 : 12);
        const ph = this._flowActorSprite.bitmap.height / (big ? 4 : 8);
        const n = actor.characterIndex();
        const sx = ((n % 4) * 3 + 1) * pw;
        const sy = (Math.floor(n / 4) * 4) * ph;
        this._flowActorSprite.setFrame(sx, sy, pw, ph);
        this._flowActorSprite.visible = true;
    };

    Scene_Battle.prototype.updateFlowActorSpriteVisibility = function() {
        if (this._flowActorSprite) {
            const actor = this._currentFlowActor;
            this._flowActorSprite.visible = !!(actor && actor.isActor() && actor.isAppeared());
        }
    };

    //-----------------------------------------------------------------------------
    // ウィンドウ関連
    //-----------------------------------------------------------------------------
    Scene_Battle.prototype.statusWindowRect = function() {
        const extra = 10; const ww = 170 * 2 + extra * 2;
        const wh = this.calcWindowHeight(3, true) - 30;
        const wx = this.isRightInputMode() ? 240 : Graphics.boxWidth - ww;
        const wy = Graphics.boxHeight - wh + extra - 4;
        return new Rectangle(wx, wy, ww, wh);
    };
    Window_BattleStatus.prototype.maxCols = function() { return 3; };
    Window_BattleStatus.prototype.drawActorFace = function() {};
    Scene_Battle.prototype.updateStatusWindowPosition = function() {};
    const _Sprite_Gauge_bitmapWidth = Sprite_Gauge.prototype.bitmapWidth;
    Sprite_Gauge.prototype.bitmapWidth = function() {
        if (SceneManager._scene instanceof Scene_Battle) return 80;
        return _Sprite_Gauge_bitmapWidth.call(this);
    };
    const _Sprite_Gauge_gaugeHeight = Sprite_Gauge.prototype.gaugeHeight;
    Sprite_Gauge.prototype.gaugeHeight = function() {
        if (SceneManager._scene instanceof Scene_Battle) return 10;
        return _Sprite_Gauge_gaugeHeight.call(this);
    };
    Sprite_Name.prototype.fontSize = function() { return 20; };
    
    //-----------------------------------------------------------------------------
    // メニュー画面関連
    //-----------------------------------------------------------------------------
    const _MenuCommandWindow = Scene_Menu.prototype.createCommandWindow;
    Scene_Menu.prototype.createCommandWindow = function() {
        _MenuCommandWindow.call(this);
        this._commandWindow.setHandler('FinishSkillList', this.commandFinishSkillList.bind(this));
    };

    const _OriginalCommands = Window_MenuCommand.prototype.addOriginalCommands;
    Window_MenuCommand.prototype.addOriginalCommands = function() {
        _OriginalCommands.call(this);
        this.addCommand('フィニッシュスキル一覧', 'FinishSkillList');
    };

    Scene_Menu.prototype.commandFinishSkillList = function() {
        SceneManager.push(Scene_FinishSkillListSimple);
    };

    function Scene_FinishSkillListSimple() {
        this.initialize(...arguments);
    }
    Scene_FinishSkillListSimple.prototype = Object.create(Scene_MenuBase.prototype);
    Scene_FinishSkillListSimple.prototype.constructor = Scene_FinishSkillListSimple;

    Scene_FinishSkillListSimple.prototype.create = function() {
        Scene_MenuBase.prototype.create.call(this);
        const leftWidth = Math.floor(Graphics.boxWidth * 0.3);
        const rightWidth = Graphics.boxWidth - leftWidth;
        const winY = 60;
        this._skillListWindow = new Window_FinishSkillList(new Rectangle(0, winY, leftWidth, Graphics.boxHeight - winY));
        this._skillListWindow.setHandler('cancel', this.popScene.bind(this));
        this._skillListWindow.setHandler('ok', this.onSkillSelect.bind(this));
        this._skillInfoWindow = new Window_FinishSkillInfo(new Rectangle(leftWidth, winY, rightWidth, Graphics.boxHeight - winY));
        this._skillListWindow.setHelpWindow(this._skillInfoWindow);
        this.addWindow(this._skillListWindow);
        this.addWindow(this._skillInfoWindow);
        this._skillListWindow.activate();
        this._skillListWindow.select(0);
        this._skillListWindow.refresh();
    };

    Scene_FinishSkillListSimple.prototype.onSkillSelect = function() {
        this._skillListWindow.activate();
    };

    function Window_FinishSkillInfo(rect) {
        Window_Help.call(this, rect);
        this.initialize(...arguments);
        this._skill = null;
    }
    Window_FinishSkillInfo.prototype = Object.create(Window_Help.prototype);
    Window_FinishSkillInfo.prototype.constructor = Window_FinishSkillInfo;
    Window_FinishSkillInfo.prototype.setItem = function(item) {
        if (item) {
            this._skill = item;
            this.refresh();
        } else {
            this.setText('');
            this._skill = null;
        }
    };

    Window_FinishSkillInfo.prototype.refresh = function() {
        this.contents.clear();
        this.resetFontSettings();
        if (!this._skill) return;
    
        let y = 0;
        const lineHeight = this.lineHeight();
        const descriptionFontSize = 20;
        const infoFontSize = 20;
    
        // スキルの説明
        this.drawText("【スキル内容】", 0, y, this.width - this.padding * 2, 'left');
        y += lineHeight;
    
        const descriptionText = `\\FS[${descriptionFontSize}]${this._skill.description.trim()}`;
        
        // ★修正箇所：yに高さを加算しないように変更
        this.drawTextEx(descriptionText, 0, y, this.width - this.padding * 2);
    
        // ★修正箇所：スキル説明の分として、yを固定値で進める
        y += lineHeight * 4;
    
        // 仮修正（上段の余白埋め。形式を戻す際に削除）
        y -= lineHeight * 2;

        // 罫線と発動条件
        y += lineHeight / 2;
        this.drawLine(y);
        y += lineHeight / 2;
    
        const flowIconMatch = this._skill.note.trim().match(/<flow_icon:\s*(\d+)(?:\s*,\s*(\d+))?(?:\s*,\s*(\d+))?>/);
        if (flowIconMatch) {
            this.drawText('【発動条件】', 0, y, this.width - this.padding * 2, 'center');
            y += lineHeight;
    
            const iconIds = flowIconMatch.slice(1).filter(id => id).map(Number);
            const iconSpacing = 8;
            const totalIconWidth = iconIds.length * 32 + (iconIds.length - 1) * iconSpacing;
            const startX = (this.width - this.padding * 2 - totalIconWidth) / 2;
            let iconX = 0;
    
            iconIds.forEach(iconId => {
                this.drawIcon(iconId, startX + iconX, y);
                iconX += 32 + iconSpacing;
            });
            y += 32 + lineHeight / 2;
        }
    
        // 罫線と追加説明
        this.drawLine(y);
        y += lineHeight / 2;
    
        const flowInfoMatch = this._skill.note.trim().match(/<flow_finish_info:([\s\S]*?)>/);
        if (flowInfoMatch) {
            const infoText = flowInfoMatch[1].trim();
            this.drawTextEx(`\\FS[${infoFontSize}]${infoText}`, 0, y, this.width - this.padding * 2);
        }
    };

    Window_FinishSkillInfo.prototype.drawLine = function(y) {
        this.contents.fillRect(this.padding, y, this.width - this.padding * 2, 2, '#808080');
    };

    function Window_FinishSkillList(rect) {
        Window_Selectable.call(this, rect);
        this.initialize(...arguments);
    }
    Window_FinishSkillList.prototype = Object.create(Window_Selectable.prototype);
    Window_FinishSkillList.prototype.constructor = Window_FinishSkillList;

    Window_FinishSkillList.prototype.maxItems = function() {
        return ($gameVariables.value(50) || []).length;
    };
    Window_FinishSkillList.prototype.itemHeight = function() { return 36; };
    Window_FinishSkillList.prototype.drawItem = function(index) {
        const skillId = ($gameVariables.value(50) || [])[index];
        const skill = $dataSkills[skillId];
        if (skill) {
            const rect = this.itemRect(index);
            this.drawText(`${index + 1}. ${skill.name}`, rect.x + 10, rect.y, rect.width - 20);
        }
    };
    Window_FinishSkillList.prototype.updateHelp = function() {
        const skillId = ($gameVariables.value(50) || [])[this.index()];
        this.setHelpWindowItem($dataSkills[skillId]);
    };
})();